package Business::OnlinePayment::Beanstream;

use strict;
use URI::Escape;
use Business::OnlinePayment;
use Business::OnlinePayment::HTTPS;
use vars qw/@ISA $VERSION $DEBUG @EXPORT @EXPORT_OK/;

@ISA=qw(Exporter AutoLoader Business::OnlinePayment::HTTPS);
@EXPORT=qw();
@EXPORT_OK=qw();
$VERSION='0.03_03';
$DEBUG = 0;

sub set_defaults{
  my $self = shift;
  $self->server('www.beanstream.com');
  $self->port('443');
  $self->path('/scripts/process_transaction.asp');

  $self->build_subs(qw( order_number avs_code merchant hash_algorithm ));
}

sub map_fields{
  my $self = shift;
  my %content = $self->content();

  my %actions = ( 'normal authorization' => 'P', 
                  'authorization only'   => 'PA',
                  'post authorization'   => 'PAC',
                  'credit'               => 'R',
                  'void'                 => 'VP',  # purchase not return
                );
  $content{action} = $actions{lc $content{action}} || $content{action};
  $content{requestType} = 'BACKEND';
  $content{expiration} = $content{exp_date}   # backward-compatibility 0.01
    if ! exists $content{expiration} && $content{exp_date};
  $content{merchant_id} = $self->merchant()  if $self->merchant();
  $self->content(%content);
}

sub remap_fields{
  my ($self,%map) = @_;
  my %content = $self->content();
  for (keys %map){ $content{$map{$_}} = $content{$_} || '' }
  $self->content(%content);
}

sub get_fields{
  my ($self,@fields) = @_;
  my %content = $self->content();
  my %new = ();

  for (@fields){ $new{$_} = $content{$_} || '' }

  return %new;
}
  

sub submit {
  my $self = shift;

  # Re: test_transaction - test mode is set on/off in the merchant account
  # settings.  No info on convenient way to set test mode per transaction.
  # For development, a developer sandbox account is available.

  if ($DEBUG > 4)  {
     my %params = $self->content;
     warn join("\n", map { "  $_ => $params{$_}" } keys %params );
  }

  $self->{_content}{owner} ||= $self->{_content}{name};
  $self->{_content}{invoice_number} ||= time;

  $self->map_fields();                 # set values with special handling
  my $loginParm = ($self->merchant() ? 'username' : 'merchant_id');
  $self->remap_fields(                 # rename keys
    login          => $loginParm,
    action         => 'trnType',
    description    => 'trnComments',
    amount         => 'trnAmount',
    invoice_number => 'trnOrderNumber',
    owner          => 'trnCardOwner',
    name           => 'ordName',
    address        => 'ordAddress1',
    city           => 'ordCity',
    state          => 'ordProvince',
    zip            => 'ordPostalCode',
    country        => 'ordCountry',
    phone          => 'ordPhoneNumber',
    email          => 'ordEmailAddress',
    card_number    => 'trnCardNumber',
    expiration     => 'trnExpYear',
    order_number   => 'adjId',
  );

  # Yes Beanstream really require phone & email, for Sales/Purchases.
  #   invoice_number/trnOrderNumber is "Recommended" per docs, or required
  #   for asynchronous transaction queries with server-to-server,
  #   BEAN_API_Integration.pdf section 13 page 53.
  # Credits/Returns/Adjustments require adjId, as well as the usual.
  my @required = qw/action amount owner name
                    address city state zip country phone email/;
  push @required, 'login'
      if exists $self->{_content}{'password'} || !$self->merchant();
  my $action = $self->{_content}{'action'};
  if ($action eq 'P' || $action eq 'PA') {  # Normal or Pre-auth [original]
    push @required, qw/card_number expiration/;
  } else {    # Capture/Return/Void [adjustment]
      push @required, 'order_number';  # invoice_number - "Recommended"
  }
  $self->required_fields(@required);
  
  # We should prepare some fields to posting, for instance ordAddress1 should be cutted and trnExpYear 
  # should be separated to trnExpMonth and trnExpYear
  
  my %content=$self->content();
  my $address = $content{ordAddress1};
  ($content{ordAddress1}, $content{ordAddress2}) = unpack 'A32 A*', $address;
  
  my $date = $content{trnExpYear};
  ($content{trnExpMonth},$content{trnExpYear}) = ($date =~/\//) ?
                                                  split /\//, $date :
                                                  unpack 'A2 A2', $date;
  
  $self->content(%content);
  
  # Now we are ready to post request
  
  my %params = $self->get_fields( qw/merchant_id username password
                                     adjId trnType trnComments
                                     trnAmount trnOrderNumber trnCardNumber
                                     trnExpYear trnExpMonth trnCardOwner
                                     ordName ordAddress1 ordCity ordProvince
                                     ordPostalCode ordCountry ordPhoneNumber
                                     ordEmailAddress requestType/ );
  if (defined $params{'password'} && !$self->merchant()) {
  # Hash Validation, BEAN_API_Integration.pdf section 14.3 p54
  # Arguably, this may depend too much on implementation of B:OP:HTTPS,
  # Net::SSLeay, LWP::UserAgent and/or HTTP::Request::Common.
    require Tie::IxHash;
    my $hashKey = delete $params{'password'};
    delete $params{'username'};
    tie %params, 'Tie::IxHash', %params;
    my $queryStr;
    #warn ($queryStr . $hashKey)  if $DEBUG > 4;
    if ($Business::OnlinePayment::HTTPS::ssl_module eq 'Net::SSLeay') {
      $queryStr = Net::SSLeay::make_form(%params);
    } else {  # LWP::UserAgent
      require URI;
      my $url = URI->new('');
      $url->query_form(%params);
      $queryStr = $url->query;
    }
    $queryStr .= $hashKey;
    warn ($queryStr)  if $DEBUG > 3;
    if (uc $self->hash_algorithm() eq 'SHA1') {
      require Digest::SHA1;
      $params{'hashValue'} = Digest::SHA1::sha1_hex($queryStr);
    } else {
      require Digest::MD5;
      $params{'hashValue'} = Digest::MD5::md5_hex($queryStr);
    }
  }

  warn join("\n", map { "  $_ => $params{$_}" } keys %params )
    if $DEBUG > 3;

  # Send transaction to Beanstream
  my ($page, $server_response, %headers) = $self->https_post( \%params );

  # Convert multi-line error to a single line.
  $server_response =~ s/[\r\n]+/ /g;
  
  # Handling server response
  if ($server_response !~ /^200/)  {
      # Connection error
      $self->is_success(0);
      my $diag_message = $server_response || "connection error";
      die $diag_message;
  }  else  {

    if ($DEBUG > 3)  {
      warn $page;  # how helpful are %headers?
    }
    $self->server_response($page);

    my %fields; 
    for my $pair (split /&/, $page) {
      my ($key, $value) = split '=', $pair;
      $fields{$key} = URI::Escape::uri_unescape($value);
      $fields{$key} =~ tr/+/ /;
    }
    warn join("\n", map { "  $_ => $fields{$_}" } keys %fields )
      if $DEBUG > 2;

    $self->result_code($fields{messageId});
    # Was messageId =~/^[129]$/, but 9 is not approved per Reporting-Guide,
    # and there are approval codes in 61..70, 561.
    if ($fields{trnApproved}) {
      $self->is_success(1);
      $self->authorization($fields{messageText});
      $self->order_number($fields{trnId});
    } else {
      $self->is_success(0);
      if ($fields{errorMessage}) {
	 $self->error_message($fields{errorMessage});
      } else {
	 $self->error_message($fields{messageText});
      }
    }

    # avs_code - Process-Transaction-API-Guide.pdf 1.6.3
    my %avsTable = (0 => '',
		    5 => 'E',
		    9 => 'E',
		    A => 'A',
		    B => 'A',
		    C => '',
		    D => 'Y',
		    E => 'E',
		    G => '',
		    I => '',
		    M => 'Y',
		    N => 'N',
		    P => 'Z',
		    R => 'R',
		    );
    $self->avs_code($avsTable{$fields{avsAddrMatch}});

  }
}

sub response_headers{
  my ($self,%headers) = @_;
  $self->{headers} = join "\n", map{"$_: $headers{$_}"} keys %headers 
                                                        if %headers;
  $self->{headers};
}

sub response_code{
  my ($self,$code) = @_;
  $self->{code} = $code if $code;
  $self->{code};
}

###
# That's all
#
1;

__END__

=head1 NAME 

Business::OnlinePayment::Beanstream - Beanstream backend for Business::OnlinePayment

=head1 SYNOPSYS

  use Business::OnlinePayment;

  # Simple case - limited to Normal Authorization & Authorization Only  
  my $tr = Business::OnlinePayment->new('Beanstream'); 
  $tr->content(
    login          => '100200000',
    action         => 'Normal Authorization',
    amount         => '1.99',
    invoice_number => '56647',
    owner          => 'John Doe',
    card_number    => '312312312312345',
    expiration     => '1219',
    name           => 'Sam Shopper',
    address        => '123 Any Street',
    city           => 'Los Angeles',
    state          => 'CA',
    zip            => '23555',
    country        => 'US',
    phone          => '123-4567',
    email          => 'Sam@shopper.com',
  );
  $tr->submit;

  if ($tr->is_success){
    print "Card processed successfully: ".$tr->authorization."\n";
  }else{
    print "Card processing was failed: ".$tr->error_message."\n";
  }

  # Hash validation - enable Credit, Post Authorization, & Void also
  my $tr = Business::OnlinePayment->new('Beanstream');  # default MD5
  my $tr = Business::OnlinePayment->new(
    'Beanstream',
    hash_algorithm => 'SHA1',
    ); 
  $tr->content(
    login          => '000000000',  # merchant ID
    password       => 'hash-key',   # Beanstream Order Settings
  # ...
  );
  $tr->submit;

  # Username/password validation - enable Credit, Post Authorization, & Void
  my $tr = Business::OnlinePayment->new(
    'Beanstream',
    merchant => '000000000',        # merchant ID
    ); 
  $tr->content(
    login          => 'user name',  # Beanstream Order Settings
    password       => 'password',   # Beanstream Order Settings
  # ...
  );
  $tr->submit;

=head1 DESCRIPTION

This module allows you to link any e-commerce order processing system directly to Beanstream transaction server (http://www.beanstream.com). All transaction fields are submitted via GET or POST to the secure transaction server at the following URL: https://www.beanstream.com/scripts/process_transaction.asp. The following fields are required:

=over 4

=item login - user name or merchant ID (Beanstream-assigned nine digit identification number)

=item action - type of transaction (Normal Authorization, Authorization Only, Post Authorization, Credit, Void)

=item amount - total order amount

=item invoice_number - the order number of the shopper's purchase

=item owner - name of the card owner

=item card_number - number of the credit card

=item expiration - expiration date formated as 'mmyy' or 'mm/yy'

=item name - name of the billing person

=item address - billing address

=item city - billing address city

=item state - billing address state/province

=item zip - billing address ZIP/postal code

=item country - billing address country

=item phone - billing contacts phone

=item email - billing contact's email

=back

Beanstream supports the following credit card:

=over 4

=item - VISA

=item - MasterCard

=item - American Express Card

=item - Discover Card

=item - JCB

=item - Diners

=back

Void Purchase is supported, but not Void Return.

For detailed information about methods see L<Business::OnlinePayment>

=head2 Note for upgrading to 0.02

As of version 0.02, this module is developed against BOP version 3.
Developer expects it to work with BOP-2 plus added HTTPS.pm (from BOP-3).

=head2 Note for upgrading to 0.03

As of version 0.03, the merchant ID is passed as a constructor parameter,
rather than a content parameter, so that the login & password content
parameters can be used for the Beanstream user name & password, which are
required for Credit (returns), Void, and Post Authorization (completion).

For backwards compatibility, in the case where only Normal Authorization
and Authorization Only are used, and the Beanstream account is configured
to not require username & password - the merchant ID is still accepted as
content login (with no password), rather than constructor parameter.

=head1 AUTHOR

Ilya Lityuga,
Randall Whitman L<whizman.com|http://whizman.com>

=head1 SEE ALSO

L<Business::OnlinePayment>

=cut
