#!perl

# Note: Place credentials for a TEST account in ./test_account file,
#       with login & password on separate lines.
#       See ./test_account_eg as example.

use strict;
use Test::More;
require "t/lib/test_account.pl";
my ($merchant, $login, $password) = test_account_or_skip();
unless ($merchant && $login) {   #  && $password
   plan skip_all => 'You must specify your merchant ID, user name, & password';
}
plan tests => 13;
use constant DEBUG => 0;

use_ok 'Business::OnlinePayment';

my ($tx, $hAlgm);
if ($merchant && $login && $password) {  # username & password
    $tx = Business::OnlinePayment->new('Beanstream',
				       merchant => $merchant);
    diag $merchant  if DEBUG>3;
    diag $login  if DEBUG>3;
} else {  # hash validation
    diag 'hash-validn'  if DEBUG>3;
    if (uc substr($login,0,5) eq 'SHA1:') {
      $password = substr($login,5);
      $hAlgm = 'SHA1';
    } elsif (uc substr($login,0,4) eq 'MD5:') {
      $password = substr($login,4);
    } else {
      $password = $login;  # implicitly MD5
    }
    $login = $merchant;
    undef $merchant;  # for $refund_tx branch
    diag $login  if DEBUG>3;
    if ($hAlgm) {
      $tx = Business::OnlinePayment->new('Beanstream',
	  hash_algorithm => 'SHA1');
    } else {
      $tx = Business::OnlinePayment->new('Beanstream');  # default hash_algm
    }
}


# Stage original purchase for refund
# avoid dup checking in case "make test" is run too close to the last run
my $amount = sprintf('%.2f', rand(10) + .5);
diag "Amount: $amount"  if DEBUG > 1;

$tx->content(
    login          => $login,
    password       => $password,
    action         => 'Normal Authorization',
    amount         => $amount,
    name           => 'Beanstream Tester',
    card_number    => '4030000010001234',
    expiration     => expiration_date(),
    address        => '123 Any Street',
    city           => 'Los Angeles',
    state          => 'CA',
    zip            => '23555',
    country        => 'US',
    phone          => '123-4567',
    email          => 'user@example.com',
);

# Account must in test mode, rather than counting on test_transaction()
$tx->test_transaction(1); # test, dont really charge

$tx->submit();

ok($tx->is_success(), 'Orig-Sale') or diag $tx->error_message;

my $auth = $tx->authorization;
my $order_number = $tx->order_number;
ok   $auth, 'authorization';
like $order_number, qr/^\d+$/, 'order number';

#warn "auth: $auth\n";
#warn "order_number: $order_number\n";

# Credit/refund of entire amount
my $refund_tx;
if ($merchant && $login && $password) {  # username & password
    $refund_tx = Business::OnlinePayment->new('Beanstream',
					      merchant => $merchant);
    diag $merchant  if DEBUG>3;
    diag $login  if DEBUG>3;
} else {  # hash validation
    diag $login  if DEBUG>3;
    $refund_tx = Business::OnlinePayment->new('Beanstream',
	  hash_algorithm => ($hAlgm || 'MD5'));
}
$refund_tx->content(
    login          => $login,
    password       => $password,
    action         => 'Credit',
    description    => 'Business::OnlinePayment visa test',
    amount         => $amount,
    authorization  => $auth,
    order_number   => $order_number,
    card_number    => '4030000010001234',
    expiration     => expiration_date(),
    name           => 'Beanstream Tester',
    address        => '123 Any Street',
    city           => 'Los Angeles',
    state          => 'CA',
    zip            => '23555',
    country        => 'US',
    phone          => '123-4567',
    email          => 'user@example.com',
);

# Account must be in test mode, rather than counting on test_transaction()
$refund_tx->test_transaction(1); # test, dont really charge
$refund_tx->submit();

ok($refund_tx->is_success(), 'Credit/Refund') or diag $refund_tx->error_message;


# Stage partial refund
$amount = sprintf('%.2f', rand(10) + 1);
diag "Amount: $amount"  if DEBUG > 1;

$tx->content(
    login          => $login,
    password       => $password,
    action         => 'Normal Authorization',
    amount         => $amount,
    name           => 'Beanstream Tester',
    card_number    => '4030000010001234',
    expiration     => expiration_date(),
    address        => '123 Any Street',
    city           => 'Los Angeles',
    state          => 'CA',
    zip            => '23555',
    country        => 'US',
    phone          => '123-4567',
    email          => 'user@example.com',
);

# Account must be in test mode, rather than counting on test_transaction()
$tx->test_transaction(1); # test, dont really charge

$tx->submit();

ok($tx->is_success(), 'Orig-Sale') or diag $tx->error_message;

$auth = $tx->authorization;
$order_number = $tx->order_number;
ok   $auth, 'authorization';
like $order_number, qr/^\d+$/, 'order number';

# Partial refund
$refund_tx->content(
    login          => $login,
    password       => $password,
    action         => 'Credit',
    description    => 'Business::OnlinePayment visa test',
    amount         => sprintf('%.2f', $amount/2),
    authorization  => $auth,
    order_number   => $order_number,
    card_number    => '4030000010001234',
    expiration     => expiration_date(),
    name           => 'Beanstream Tester',
    address        => '123 Any Street',
    city           => 'Los Angeles',
    state          => 'CA',
    zip            => '23555',
    country        => 'US',
    phone          => '123-4567',
    email          => 'user@example.com',
);

# Account must be in test mode, rather than counting on test_transaction()
$refund_tx->test_transaction(1); # test, dont really charge
$refund_tx->submit();

ok($refund_tx->is_success(), 'Partial Refund') or diag $refund_tx->error_message;


# Stage invalid refund
$amount = sprintf('%.2f', rand(10) + .5);
diag "Amount: $amount"  if DEBUG > 1;

$tx->content(
    login          => $login,
    password       => $password,
    action         => 'Normal Authorization',
    amount         => $amount,
    name           => 'Beanstream Tester',
    card_number    => '4030000010001234',
    expiration     => expiration_date(),
    address        => '123 Any Street',
    city           => 'Los Angeles',
    state          => 'CA',
    zip            => '23555',
    country        => 'US',
    phone          => '123-4567',
    email          => 'user@example.com',
);

# Account must in test mode, rather than counting on test_transaction()
$tx->test_transaction(1); # test, dont really charge

$tx->submit();

ok($tx->is_success(), 'Orig-Sale') or diag $tx->error_message;

$auth = $tx->authorization;
$order_number = $tx->order_number;
ok   $auth, 'authorization';
like $order_number, qr/^\d+$/, 'order number';

# Invalid refund amount - greater than original purchase
$refund_tx->content(
    login          => $login,
    password       => $password,
    action         => 'Credit',
    description    => 'Business::OnlinePayment visa test',
    amount         => sprintf('%.2f', 2 * $amount),
    authorization  => $auth,
    order_number   => $order_number,
    card_number    => '4030000010001234',
    expiration     => expiration_date(),
    name           => 'Beanstream Tester',
    address        => '123 Any Street',
    city           => 'Los Angeles',
    state          => 'CA',
    zip            => '23555',
    country        => 'US',
    phone          => '123-4567',
    email          => 'user@example.com',
);

# Account must in test mode, rather than counting on test_transaction()
$refund_tx->test_transaction(1); # test, dont really charge
$refund_tx->submit();

ok($refund_tx->error_message, 'Invalid Refund');
diag ($refund_tx->error_message || 'no error')  if DEBUG > 1;
